// Line Tracer Robot Kit TypeR V2.0 2025 code.
// copyright RoBoTeNa Co., Ltd.
#include <MsTimer2.h>            // タイマー割り込みを利用する為に必要なファイル

#include <Wire.h>                // MPU-6050 Accelerometer + Gyro
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
#include <Adafruit_MPU6050.h>
#include <Adafruit_Sensor.h>

/*================== マクロ定義 ==================*/
#define LED_PIN        13
#define PUSHSW1_PIN     0
#define PUSHSW2_PIN     1
#define LINE1_PIN      A0
#define LINE2_PIN      A1
#define LINE3_PIN      A2
#define LINE4_PIN      A3
#define LINE5_PIN      A6
#define BATTVOLT_PIN   A7
#define ANGLE_PIN      A7
#define MARKER_L_PIN    7
#define MARKER_R_PIN    8
#define BUZZER_PIN      4
#define ENC_PIN         2
#define MOTOR_L_IN1     5
#define MOTOR_L_IN2     6
#define MOTOR_R_IN1     9
#define MOTOR_R_IN2    10
#define ON              1
#define OFF             0
#define PWM_LIMIT     255 //0-255

/* ライン白黒判定閾値設定 */
#define LINE1_THRESHOLD 800
#define LINE2_THRESHOLD 800
#define LINE3_THRESHOLD 800
#define LINE4_THRESHOLD 800
#define LINE5_THRESHOLD 800

// Declaration for an SSD1306 display connected to I2C (SDA, SCL pins)
#define SCREEN_WIDTH 128 // OLED display width, in pixels
#define SCREEN_HEIGHT 64 // OLED display height, in pixels
#define OLED_RESET     4 // Reset pin # (or -1 if sharing Arduino reset pin)
#define MPU6050_GZ  0x47

/*================== 変数宣言 ==================*/
unsigned char Pattern = 0;            // 動作パターン（スタート前:0 ,通常走行:11）

/* センサ関連 */
int iSensorPattern;                   // センサ状態保持用
unsigned char LineGet = 0x00;         // ラインセンサの状態
unsigned char MarkerGet = 0x00;       // マーカーセンサの状態
unsigned int  A_Line1, A_Line2, A_Line3, A_Line4, A_Line5;
unsigned char Speed, St, Co;

/* タイマ関連 */
unsigned long Cnt_1ms;                // 時間計測用カウンタ
unsigned long Cnt_Total;              // 総合計時間計測用カウンタ
unsigned int  Cnt_Gyro;               // ジャイロ計測間隔カウンタ
unsigned int  Cnt_Marker;             // マーカーチャタリング対策カウンタ

/* エンコーダ関連 */
unsigned int  EncCnt;                 // 1m/sで10になる
unsigned int  EncBuff;                // 計算用　割り込み内で使用
unsigned long EncTotal;               // 積算値保存用(1m走行で500パルス)
unsigned long EncGoal;                // ゴール距離測定用

/*  サーボ関連 */
int iSensorBefore;                    // 前回のセンサ値保存
int iServoPwm;                        // サーボＰＷＭ値
unsigned char Cross_Mode;             // 交差点モードフラグ

/* ジャイロセンサ関連 */
const int MPU6050_addr = 0x68;
int16_t AccX, AccY, AccZ, Temp, GyroX, GyroY, GyroZ;
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

/*================== 初期設定 ==================*/
void setup() {
  TCCR0B = (TCCR0B & 0b11111000) | 0x01;
  TCCR1B = (TCCR1B & 0b11111000) | 0x01;
  TCCR2B = (TCCR2B & 0b11111000) | 0x01;  

  Serial.begin(9600);       // set up Serial library at 9600 bps
  Serial.println("Tracer Robot!");
  pinMode(LED_PIN, OUTPUT);
  pinMode(BUZZER_PIN, OUTPUT);
  
  pinMode(MOTOR_L_IN1, OUTPUT);
  pinMode(MOTOR_L_IN2, OUTPUT);
  pinMode(MOTOR_R_IN1, OUTPUT);
  pinMode(MOTOR_R_IN2, OUTPUT);
  
  // initialize the pushbutton pin as an input:
  pinMode(PUSHSW1_PIN, INPUT_PULLUP);
  pinMode(PUSHSW2_PIN, INPUT_PULLUP);
  pinMode(MARKER_L_PIN, INPUT_PULLUP);
  pinMode(MARKER_R_PIN, INPUT_PULLUP);

  // Timer2割込設定
  MsTimer2::set(1, Timer2_Int);      // 1ms毎に Timer2_Int 割込み
  MsTimer2::start();                 // タイマー割り込み開始

  // エンコーダ設定
  pinMode(ENC_PIN,INPUT);
  attachInterrupt(0, Enc_changed, CHANGE); //信号変化に合わせて割り込み処理

  // OLED表示関連
  if (!display.begin(SSD1306_SWITCHCAPVCC, 0x3C)) { //Ive changed the address //already chill
    Serial.println(F("SSD1306 allocation failed"));
    for (;;); // Don't proceed, loop forever
  }
  display.display();
  delay(2000); // Pause for 2 seconds

  //ジャイロセンサ関連
  Wire.begin();
  Wire.setClock(400000L); // set I2C to a fast mode (400kpbs)
  Wire.beginTransmission(MPU6050_addr);
  Wire.write(0x6B);
  Wire.write(0);
  Wire.endTransmission(true);
  display.clearDisplay();
  display.setTextSize(1);
  display.setTextColor(WHITE);
  display.setRotation(0);

  // ブザー初期化
  Buzzer(OFF);

}

/*================== メインループ ==================*/
void loop() {
  LineGet = LineSens_inp();     //ラインデジタルセンサ値読み出し
  MarkerGet = MarkerSens_inp(); //マーカーデジタルセンサ値読み出し

  if( Cnt_Gyro > 10 ){          // 10msに1回ジャイロ計測値取得
      GyroReadZ();              // ジャイロセンサ値読み出し
      Cnt_Gyro = 0;
  }

  /*=== ジャイロによる直線・コーナー判定処理 ===*/
  if( (GyroZ>100)||(GyroZ<-100) ){ /* 角速度30以上ならコーナー判定 */
    Speed = Co;                 // コーナー走行速度
  } else {
    Speed = St;                 // 直線走行速度
  }
    
  switch( Pattern ) {
  case 0://スタート待ち
    MotorLR( 0, 0 );
    if( digitalRead(PUSHSW1_PIN) == 0 ){
      Cnt_1ms = 0;
      Cnt_Total = 0;
      initialize();                       // 初期化実行
      Pattern = 11;                       // case:11通常トレースへ遷移
    }
    if( digitalRead(PUSHSW2_PIN) == 0 ){
      Cnt_1ms = 0;
      Cnt_Total = 0;
      initialize();                       // 初期化実行
      Pattern = 7;                        // case:7宴会芸モードへ遷移
    }
    /*=== ラインセンサ値に応じて速度設定（1m/s=10）===*/
    if( LineGet==0b00000001 ){
        St = 6;
        Co = 4;
    } else if( LineGet==0b00000010 ){
        St = 8;
        Co = 6;        
    } else if( LineGet==0b00001000 ){
        St = 12;
        Co = 10;        
    } else if( LineGet==0b00010000 ){
        St = 14;
        Co = 12;        
    } else {
        St = 10;
        Co = 8;
    }

    // 各種センサ値をOLEDに表示
    display.clearDisplay();
    display.setCursor(0, 0);
    display.print("A0: ");
    display.print(A_Line1);
    display.setCursor(50, 0);
    display.print("A1: ");
    display.print(A_Line2);
    display.setCursor(0, 10);
    display.print("A2: ");
    display.print(A_Line3);
    display.setCursor(50, 10);
    display.print("A3: ");
    display.print(A_Line4);
    display.setCursor(0, 20);
    display.print("A6: ");
    display.print(A_Line5);
    display.setCursor(0, 30);
    display.print("EncTotal: ");
    display.print(EncTotal);
    display.setCursor(0, 40);
    display.print("GyroZ: ");
    display.print(GyroZ);
    display.setCursor(0, 50);
    display.print("St:");
    display.print(St);
    display.setCursor(50, 50);
    display.print("Co:");
    display.print(Co);
    display.display();
    break;

  case 1://停止
    MotorLR( 0, 0 );
    if( Cnt_1ms > 5000 ){ //5s経過後に停止パターンへ移行
        Cnt_1ms = 0;
        Pattern = 0;
    }
    break;

  case 7:// 宴会芸
    if( (GyroZ>10) || (GyroZ<-10) ){
      MotorLR( GyroZ, -GyroZ );
    } else {
      MotorLR( 0, 0 );
    }
    break;

  case 9:// モータテスト
    MotorLR( 0, 100 );
    break;

  case 10:// ラインセンサ値確認
    MotorLR( 0, 0 );
    Serial.println("LINE1=");
    Serial.println(A_Line1);
    Serial.println("LINE2=");
    Serial.println(A_Line2);
    Serial.println("LINE3=");
    Serial.println(A_Line3);
    Serial.println("LINE4=");
    Serial.println(A_Line4);
    Serial.println("LINE5=");
    Serial.println(A_Line5);
    break;

  case 11:// 通常トレース 
    NomalTrace();                     // 通常ライントレース処理
    
    /*=== ゴールマーカー検出処理 ===*/
    if( ((MarkerGet == 0x01) && (Cnt_Total > 2000)) && (Cnt_1ms > 100)){
      Cnt_1ms = 0;
      EncGoal = EncTotal;
      Pattern = 21;
    }
    /*=== コーナーマーカー＆交差点検出処理 ===*/
    if( (MarkerGet == 0x02) || (MarkerGet == 0x03)){
      Cnt_1ms = 0;
    }
    break;

  case 21:// ゴール検出処理
    NomalTrace();

    if( (MarkerGet == 0x02)||(MarkerGet == 0x03) ){ // 左マーカー検出
        Cnt_1ms = 0;
        Pattern = 11;
    }
    if( (EncTotal-EncGoal) > 250 ){     // 500mm走行後に停止パターンへ移行
        Cnt_1ms = 0;
        Pattern = 1;
    }
    break;

  default:
    break;
  }
}

/************************************************************************
 初期化処理
************************************************************************/
void initialize(){
    //タイマ、モード関連
    Cnt_1ms = 0;
    Cnt_Gyro = 0;
    iSensorBefore = 0;
    iServoPwm = 0;               // サーボＰＷＭ値
    iSensorPattern = 0;          // センサ状態保持用
  
    //エンコーダ関連変数初期化
    EncCnt = 0;
    EncBuff = 0;
    EncTotal = 0;

}

/************************************************************************
 TIMER2割込処理
************************************************************************/
void Timer2_Int()
{
  static int cnt100;
  static int cnt20;
  unsigned int i = 0;
  
  Cnt_1ms++;    //時間計測用
  Cnt_Total++;  //合計時間計測用
  Cnt_Gyro++;   //ジャイロ計測間隔用
  Cnt_Marker++; //マーカーセンサチャタリング対策用
  
  analog_inp();                        // アナログ値取得
  PD_Control();                        // サーボモータ制御
 
  // 100ms周期の処理
  cnt100++;
  if( cnt100 > 100 ){
      if( Pattern == 0 ){
          static boolean output = HIGH;  // プログラム起動前に１回だけHIGH(1)で初期化される
          digitalWrite(13, output);      // 13番ピン(LED)に出力する(HIGH>ON LOW>OFF)
          output = !output;              // 現在のoutput内容を反転(HIGH→LOW/LOW→HIGH)させoutputにセットする
      }
      cnt100 = 0;
  }
  // 20ms周期の処理
  cnt20++;
  if( cnt20 > 20 ){
      EncCnt = EncBuff;
      EncTotal += EncCnt;
      EncBuff = 0;
      cnt20 = 0;
  }
}

/************************************************************************
 スピード制御
************************************************************************/
int EncPwm(int enc){

    int pwm;
    
    if( EncCnt > ( enc + 5 ) ) {
        pwm = -255;
    } else if( EncCnt > ( enc + 4 ) ) {
        pwm = -150;
    } else if( EncCnt > ( enc + 3 ) ) {
        pwm = -100;
    } else if( EncCnt > ( enc + 2 ) ) {
        pwm = -50;
    } else if( EncCnt > enc ) {
        pwm = 0;
    } else if( EncCnt < ( enc - 5 ) ) {
        pwm = 255;
    } else if( EncCnt < ( enc - 3 ) ) {
        pwm = 150;
    } else if( EncCnt < ( enc - 2 ) ) {
        pwm = 100;
    } else {
        pwm = 80;
    }
    
    return pwm;
}

/************************************************************************
 通常トレース
************************************************************************/
void NomalTrace( void ){

    MotorLR( (iServoPwm+EncPwm(Speed)), (-iServoPwm+EncPwm(Speed)) );
}

/************************************************************************
 モータ制御
************************************************************************/
void MotorLR(int l, int r){

    //最大値を超えない様にする処理
    if( l > PWM_LIMIT ){
      l = PWM_LIMIT;
    }
    if( r > PWM_LIMIT ){
      r = PWM_LIMIT;
    }
    if( l < -PWM_LIMIT ){
      l = -PWM_LIMIT;
    }
    if( r < -PWM_LIMIT ){
      r = -PWM_LIMIT;
    }

  //左モータ
    if (l == 0){
      digitalWrite(MOTOR_L_IN1, HIGH);
      digitalWrite(MOTOR_L_IN2, HIGH);
    } else if (l < 0){
      l = -l;
      l = 255-l;
      digitalWrite(MOTOR_L_IN1, HIGH);
      analogWrite(MOTOR_L_IN2, l);
    } else if (l > 0){
      l = 255-l;
      analogWrite(MOTOR_L_IN1, l);
      digitalWrite(MOTOR_L_IN2, HIGH);
    }
    
  //右モータ
    if (r == 0){
      digitalWrite(MOTOR_R_IN1, HIGH);
      digitalWrite(MOTOR_R_IN2, HIGH);
    } else if (r < 0){
      r = -r;
      r = 255-r;
      digitalWrite(MOTOR_R_IN1, HIGH);
      analogWrite(MOTOR_R_IN2, r);
    } else if (r > 0){
      r = 255-r;
      analogWrite(MOTOR_R_IN1, r);
      digitalWrite(MOTOR_R_IN2, HIGH);      
    }
}

/************************************************************************
 エンコーダ割込処理
************************************************************************/
void Enc_changed()
{
    EncBuff++;
}

/************************************************************************
 AD値読み込み
************************************************************************/
void analog_inp( void )
{
    A_Line1 = analogRead(LINE1_PIN);
    A_Line2 = analogRead(LINE2_PIN);
    A_Line3 = analogRead(LINE3_PIN);
    A_Line4 = analogRead(LINE4_PIN);
    A_Line5 = analogRead(LINE5_PIN);
}

/************************************************************************
 ライン用アナログセンサ値読み込み＆デジタル変換
************************************************************************/
unsigned long LineSens_inp( void )
{
    unsigned char sensor = 0;

  if( A_Line1 > LINE1_THRESHOLD ){
      sensor += 1;
  }
  if( A_Line2 > LINE2_THRESHOLD ){
      sensor += 2;
  }
  if( A_Line3 > LINE3_THRESHOLD ){
      sensor += 4;
  } 
  if( A_Line4 > LINE4_THRESHOLD ){
      sensor += 8;
  }
  if( A_Line5 > LINE5_THRESHOLD ){
      sensor += 16;
  }
    return sensor;
}

/************************************************************************
 マーカー用デジタルセンサ値読み込み
************************************************************************/
unsigned char MarkerSens_inp( void )
{
    unsigned char marker = 0;

    if( digitalRead(MARKER_R_PIN) == 1 ){
        marker += 1;
    }
    if( digitalRead(MARKER_L_PIN) == 1 ){
        marker += 2;
    }

    return marker;
}

/************************************************************************
 コースアウト緊急停止制御
************************************************************************/
void CourseOut_check()
{
    static unsigned int cnt;

    if( (LineGet==0x00 || LineGet==0x0f) && (Pattern==11) ){
        cnt++;
    } else {
        cnt = 0;
    }
    if( cnt > 2000 ){ //全点灯or全消灯が2000ms以上続いたら停止
        Pattern = 1;
    }
}

/************************************************************************
 ブザー制御
************************************************************************/
void Buzzer(int beep){
   digitalWrite(BUZZER_PIN, beep);
}

/************************************************************************
 アナログセンサ値取得
************************************************************************/
int getAnalogSensor( void )
{
    int ret;

    ret = A_Line2 - A_Line4;                    /* アナログセンサ情報取得       */

    switch( iSensorPattern ) {
    case 0:
        if( LineSens_inp() == 0x01 ) {
            ret = 800;
            iSensorPattern = 1;
            break;
        }
        if( LineSens_inp() == 0x10 ) {
            ret = -800;
            iSensorPattern = 2;
            break;
        }
        break;

    case 1:
        /* センサ右寄り */
        ret = 800;
        if( (LineSens_inp() == 0x04)||(LineSens_inp() == 0x02) ) {
            iSensorPattern = 0;
        }
        break;

    case 2:
        /* センサ左寄り */
        ret = -800;
        if( (LineSens_inp() == 0x04)||(LineSens_inp() == 0x08)) {
            iSensorPattern = 0;
        }
        break;
    }

    /* クロスラインでの引き込み防止 */
    if( (A_Line2 > 800) && (A_Line4 > 800) ) {
        ret = 0;
    }

    return ret;
}

/************************************************************************
 PD制御
************************************************************************/
void PD_Control( void )
{
    int i, iRet, iP, iD;
    int kp, kd;

    i = getAnalogSensor();              /* センサ値取得           */
    kp = 14;                            /* 調整できたらP,D値は固定 */
    kd = 150;

    /* サーボモータ用PWM値計算 */
    iP = kp * i;                        /* 比例                  */
    iD = kd * (iSensorBefore - i );     /* 微分(目安はPの5～10倍)  */
    iRet = iP - iD;
    iRet /= 64;

    /* PWMの上限の設定 */
    if( iRet >  255 ) iRet =  255;
    if( iRet < -255 ) iRet = -255;
    iServoPwm = iRet;

    iSensorBefore = i;                  /* 次回はこの値が1ms前の値となる*/
}

/************************************************************************
 MPU6050 Read処理
************************************************************************/
void GyroReadZ(void )
{
    Wire.beginTransmission(MPU6050_addr);
    Wire.write(0x47);
    Wire.endTransmission(false);
    Wire.requestFrom(MPU6050_addr, 14, true);
    GyroZ = Wire.read() << 8 | Wire.read();
    GyroZ = (GyroZ / 100);
}
